'use strict';

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

/* --------------------------------------------------------------
 overview_settings_modal_controller.js 2017-12-06
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.overview_settings_modal_controller = jse.libs.overview_settings_modal_controller || {};

/**
 * Overview settings modal controller class.
 *
 * @module Admin/Libs/overview_settings_modal
 * @exports jse.libs.overview_settings_modal
 */
(function (exports) {
    /**
     * Class representing a controller for the orders overview settings modal.
     */
    var OverviewSettingsModalController = function () {
        /**
         * Creates an instance of OrdersOverviewSettingsModalController.
         *
         * @param {jQuery}    $element              Module element.
         * @param {Object}    userCfgService        User configuration service library.
         * @param {Object}    loadingSpinner        Loading spinner library.
         * @param {Number}    userId                ID of currently signed in user.
         * @param {String}    defaultColumnSettings Default column settings.
         * @param {Object}    translator            Translator library.
         * @param {String}    page                  Page name (e.g.: 'orders', 'invoices').
         */
        function OverviewSettingsModalController($element, userCfgService, loadingSpinner, userId, defaultColumnSettings, translator, page) {
            _classCallCheck(this, OverviewSettingsModalController);

            // Elements
            this.$element = $element;
            this.$submitButton = $element.find('button.submit-button');
            this.$settings = $element.find('ul.settings');
            this.$modal = $element.parents('.modal');
            this.$modalFooter = $element.find('.modal-footer');
            this.$resetDefaultLink = $element.find('a.reset-action');

            // Loading spinner
            this.$spinner = null;

            // Selector strings
            this.sortableHandleSelector = 'span.sort-handle';
            this.rowHeightValueSelector = 'select#setting-value-row-height';
            this.displayTooltipValueSelector = 'input#setting-value-display-tooltips';

            // Class names
            this.errorMessageClassName = 'error-message';
            this.loadingClassName = 'loading';

            // Libraries
            this.userCfgService = userCfgService;
            this.loadingSpinner = loadingSpinner;
            this.translator = translator;

            // Prefixes
            this.settingListItemIdPrefix = 'setting-';
            this.settingValueIdPrefix = 'setting-value-';

            // User configuration keys
            this.CONFIG_KEY_COLUMN_SETTINGS = page + 'OverviewSettingsColumns';
            this.CONFIG_KEY_ROW_HEIGHT_SETTINGS = page + 'OverviewSettingsRowHeight';
            this.CONFIG_KEY_DISPLAY_TOOLTIPS_SETTINGS = page + 'OverviewSettingsDisplayTooltips';

            // Default values
            this.DEFAULT_ROW_HEIGHT_SETTING = 'large';
            this.DEFAULT_COLUMN_SETTINGS = defaultColumnSettings.split(',');
            this.DEFAULT_DISPLAY_TOOLTIPS_SETTINGS = 'true';

            // ID of currently signed in user.
            this.userId = userId;
        }

        /**
         * Binds the event handlers.
         *
         * @return {OverviewSettingsModalController} Same instance for method chaining.
         */


        _createClass(OverviewSettingsModalController, [{
            key: 'initialize',
            value: function initialize() {
                var _this = this;

                // Attach event handler for click action on the submit button.
                this.$submitButton.on('click', function (event) {
                    return _this._onSubmitButtonClick(event);
                });

                // Attach event handler for click action on the reset-default link.
                this.$resetDefaultLink.on('click', function (event) {
                    return _this._onResetSettingsLinkClick(event);
                });

                // Attach event handlers to modal.
                this.$modal.on('show.bs.modal', function (event) {
                    return _this._onModalShow(event);
                }).on('shown.bs.modal', function (event) {
                    return _this._onModalShown(event);
                });

                return this;
            }

            /**
             * Fades out the modal content.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onModalShow',
            value: function _onModalShow() {
                this.$element.addClass(this.loadingClassName);

                return this;
            }

            /**
             * Updates the settings, clears any error messages and initializes the sortable plugin.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onModalShown',
            value: function _onModalShown() {
                this._refreshSettings()._clearErrorMessage()._initSortable();

                return this;
            }

            /**
             * Activates the jQuery UI Sortable plugin on the setting list items element.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_initSortable',
            value: function _initSortable() {
                // jQuery UI Sortable plugin options.
                var options = {
                    items: '> li',
                    axis: 'y',
                    cursor: 'move',
                    handle: this.sortableHandleSelector,
                    containment: 'parent'
                };

                // Activate sortable plugin.
                this.$settings.sortable(options).disableSelection();

                return this;
            }

            /**
             * Returns a sorted array containing the IDs of all activated settings.
             *
             * @return {Array}
             *
             * @private
             */

        }, {
            key: '_serializeColumnSettings',
            value: function _serializeColumnSettings() {
                var _this2 = this;

                // Map iterator function to remove the 'setting-' prefix from list item ID.
                var removePrefixIterator = function removePrefixIterator(item) {
                    return item.replace(_this2.settingListItemIdPrefix, '');
                };

                // Filter iterator function, to accept only list items with activated checkboxes.
                var filterIterator = function filterIterator(item) {
                    return _this2.$settings.find('#' + _this2.settingValueIdPrefix + item).is(':checked');
                };

                // Return array with sorted, only active columns.
                return this.$settings.sortable('toArray').map(removePrefixIterator).filter(filterIterator);
            }

            /**
             * Returns the value of the selected row height option.
             *
             * @return {String}
             *
             * @private
             */

        }, {
            key: '_serializeRowHeightSetting',
            value: function _serializeRowHeightSetting() {
                return this.$element.find(this.rowHeightValueSelector).val();
            }

            /**
             * Returns the value of the selected tooltip display option.
             *
             * @return {String}
             *
             * @private
             */

        }, {
            key: '_serializeDisplayTooltipSetting',
            value: function _serializeDisplayTooltipSetting() {
                return this.$element.find(this.displayTooltipValueSelector).prop('checked');
            }

            /**
             * Shows the loading spinner, saves the settings to the user configuration,
             * closes the modal to finally re-render the datatable.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onSubmitButtonClick',
            value: function _onSubmitButtonClick() {
                var _this3 = this;

                // Retrieve setting values.
                var columnSettings = this._serializeColumnSettings();
                var rowHeightSetting = this._serializeRowHeightSetting();
                var displayTooltipSetting = this._serializeDisplayTooltipSetting();

                // Remove any error message and save settings.
                this._toggleLoadingSpinner(true)._clearErrorMessage()._saveColumnSettings(columnSettings).then(function () {
                    return _this3._saveDisplayTooltipSetting(displayTooltipSetting);
                }).then(function () {
                    return _this3._saveRowHeightSetting(rowHeightSetting);
                }).then(function () {
                    return _this3._onSaveSuccess();
                }).catch(function () {
                    return _this3._onSaveError();
                });

                return this;
            }

            /**
             * Prevents the browser to apply the default behavoir and
             * resets the column order and row size to the default setting values.
             *
             * @param {jQuery.Event} event Fired event.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onResetSettingsLinkClick',
            value: function _onResetSettingsLinkClick(event) {
                // Prevent default behavior.
                event.preventDefault();
                event.stopPropagation();

                // Reset to default settings.
                this._setDefaultSettings();

                return this;
            }

            /**
             * Shows and hides the loading spinner.
             *
             * @param {Boolean} doShow Show the loading spinner?
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             */

        }, {
            key: '_toggleLoadingSpinner',
            value: function _toggleLoadingSpinner(doShow) {
                if (doShow) {
                    // Fade out modal content.
                    this.$element.addClass(this.loadingClassName);

                    // Show loading spinner.
                    this.$spinner = this.loadingSpinner.show(this.$element);

                    // Fix spinner z-index.
                    this.$spinner.css({ 'z-index': 9999 });
                } else {
                    // Fade out modal content.
                    this.$element.removeClass(this.loadingClassName);

                    // Hide the loading spinner.
                    this.loadingSpinner.hide(this.$spinner);
                }

                return this;
            }

            /**
             * Handles the behavior on successful setting save action.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onSaveSuccess',
            value: function _onSaveSuccess() {
                window.location.reload();
                return this;
            }

            /**
             * Removes any error message, if found.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_clearErrorMessage',
            value: function _clearErrorMessage() {
                // Error message.
                var $errorMessage = this.$modalFooter.find('.' + this.errorMessageClassName);

                // Remove if it exists.
                if ($errorMessage.length) {
                    $errorMessage.remove();
                }

                return this;
            }

            /**
             * Handles the behavior on thrown error while saving settings.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_onSaveError',
            value: function _onSaveError() {
                // Error message.
                var errorMessage = this.translator.translate('TXT_SAVE_ERROR', 'admin_general');

                // Define error message element.
                var $error = $('<span/>', { class: this.errorMessageClassName, text: errorMessage });

                // Hide the loading spinner.
                this._toggleLoadingSpinner(false);

                // Add error message to modal footer.
                this.$modalFooter.prepend($error).hide().fadeIn();

                return this;
            }

            /**
             * Returns the configuration value for the column settings.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_getColumnSettings',
            value: function _getColumnSettings() {
                // Configuration data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_COLUMN_SETTINGS
                };

                // Request data from user configuration service.
                return this._getFromUserCfgService(data);
            }

            /**
             * Returns the configuration value for the row heights.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_getRowHeightSetting',
            value: function _getRowHeightSetting() {
                // Configuration data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_ROW_HEIGHT_SETTINGS
                };

                // Request data from user configuration service.
                return this._getFromUserCfgService(data);
            }

            /**
             * Returns the configuration value for the tooltip display option.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_getDisplayTooltipSetting',
            value: function _getDisplayTooltipSetting() {
                // Configuration data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_DISPLAY_TOOLTIPS_SETTINGS
                };

                // Request data from user configuration service.
                return this._getFromUserCfgService(data);
            }

            /**
             * Returns the value for the passed user configuration data.
             *
             * @param {Object} data                   User configuration data.
             * @param {Number} data.userId            User ID.
             * @param {String} data.configurationKey  User configuration key.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_getFromUserCfgService',
            value: function _getFromUserCfgService(data) {
                var _this4 = this;

                // Promise handler.
                var handler = function handler(resolve, reject) {
                    // User configuration service request options.
                    var options = {
                        onError: function onError() {
                            return reject();
                        },
                        onSuccess: function onSuccess(response) {
                            return resolve(response.configurationValue);
                        },
                        data: data
                    };

                    // Get configuration value.
                    _this4.userCfgService.get(options);
                };

                return new Promise(handler);
            }

            /**
             * Saves the data via the user configuration service.
             *
             * @param {Object} data                     User configuration data.
             * @param {Number} data.userId              User ID.
             * @param {String} data.configurationKey    User configuration key.
             * @param {String} data.configurationValue  User configuration value.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_setWithUserCfgService',
            value: function _setWithUserCfgService(data) {
                var _this5 = this;

                // Promise handler.
                var handler = function handler(resolve, reject) {
                    // User configuration service request options.
                    var options = {
                        onError: function onError() {
                            return reject();
                        },
                        onSuccess: function onSuccess(response) {
                            return resolve();
                        },
                        data: data
                    };

                    // Set configuration value.
                    _this5.userCfgService.set(options);
                };

                return new Promise(handler);
            }

            /**
             * Saves the column settings via the user configuration service.
             *
             * @param {String[]} columnSettings Sorted array with active column.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_saveColumnSettings',
            value: function _saveColumnSettings(columnSettings) {
                // Check argument.
                if (!columnSettings || !Array.isArray(columnSettings)) {
                    throw new Error('Missing or invalid column settings');
                }

                // User configuration request data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_COLUMN_SETTINGS,
                    configurationValue: JSON.stringify(columnSettings)
                };

                // Save via user configuration service.
                return this._setWithUserCfgService(data);
            }

            /**
             * Saves the row height setting via the user configuration service.
             *
             * @param {String} rowHeightSetting Value of the selected row height setting.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_saveRowHeightSetting',
            value: function _saveRowHeightSetting(rowHeightSetting) {
                // Check argument.
                if (!rowHeightSetting || typeof rowHeightSetting !== 'string') {
                    throw new Error('Missing or invalid row height setting');
                }

                // User configuration request data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_ROW_HEIGHT_SETTINGS,
                    configurationValue: rowHeightSetting
                };

                // Save via user configuration service.
                return this._setWithUserCfgService(data);
            }

            /**
             * Saves the display tooltip setting via the user configuration service.
             *
             * @param {String} displayTooltipSetting Value.
             *
             * @return {Promise}
             *
             * @private
             */

        }, {
            key: '_saveDisplayTooltipSetting',
            value: function _saveDisplayTooltipSetting(displayTooltipSetting) {
                // User configuration request data.
                var data = {
                    userId: this.userId,
                    configurationKey: this.CONFIG_KEY_DISPLAY_TOOLTIPS_SETTINGS,
                    configurationValue: displayTooltipSetting
                };

                // Save via user configuration service.
                return this._setWithUserCfgService(data);
            }

            /**
             * Retrieves the saved setting configuration and reorders/updates the settings.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_refreshSettings',
            value: function _refreshSettings() {
                var _this6 = this;

                // Show loading spinner.
                this._toggleLoadingSpinner(true);

                // Error handler function to specify the behavior on errors while processing.
                var onRefreshSettingsError = function onRefreshSettingsError(error) {
                    // Output warning.
                    console.warn('Error while refreshing', error);

                    // Hide the loading spinner.
                    _this6._toggleLoadingSpinner(false);
                };

                // Remove any error message, set row height,
                // reorder and update the settings and hide the loading spinner.
                this._clearErrorMessage()._getRowHeightSetting().then(function (rowHeightValue) {
                    return _this6._setRowHeight(rowHeightValue);
                }).then(function () {
                    return _this6._getDisplayTooltipSetting();
                }).then(function (displayTooltipSetting) {
                    return _this6._setDisplayTooltipSetting(displayTooltipSetting);
                }).then(function () {
                    return _this6._getColumnSettings();
                }).then(function (columnSettings) {
                    return _this6._setColumnSettings(columnSettings);
                }).then(function () {
                    return _this6._toggleLoadingSpinner(false);
                }).catch(onRefreshSettingsError);

                return this;
            }

            /**
             * Sets the row height setting value.
             *
             * @param {String} value Row height value.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_setRowHeight',
            value: function _setRowHeight() {
                var value = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.DEFAULT_ROW_HEIGHT_SETTING;

                this.$element.find(this.rowHeightValueSelector).val(value);

                return this;
            }

            /**
             * Sets the display tooltips setting value.
             *
             * @param {String} value Display tooltips value.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_setDisplayTooltipSetting',
            value: function _setDisplayTooltipSetting() {
                var value = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.DEFAULT_DISPLAY_TOOLTIPS_SETTINGS;

                this.$element.find(this.displayTooltipValueSelector).prop('checked', value === 'true').trigger('change');

                return this;
            }

            /**
             * Reorders and updates the column setting values.
             *
             * @param {String|Array} columnSettings Stringified JSON array containing the saved column settings.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_setColumnSettings',
            value: function _setColumnSettings() {
                var _this7 = this;

                var columnSettings = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.DEFAULT_COLUMN_SETTINGS;

                // Regex for escape character.
                var ESCAPE_CHAR = /\\/g;

                // No need to parse from JSON on default value as it is an array.
                if (!Array.isArray(columnSettings)) {
                    // Remove escape characters from and parse array from JSON.
                    columnSettings = columnSettings.replace(ESCAPE_CHAR, '');
                    columnSettings = JSON.parse(columnSettings);
                }

                // Cache container to temporarily hold all active list items in sorted order.
                // The children of this element will be prepended to the setting list item container to retain the
                // sorting order.
                var $sortedItems = $('<div/>');

                // Iterator function to prepend active list items to the top and activate the checkbox.
                var settingIterator = function settingIterator(setting) {
                    // List item ID.
                    var id = _this7.settingListItemIdPrefix + setting;

                    // Affected setting list item.
                    var $listItem = _this7.$settings.find('#' + id);

                    // Checkbox of affected list item.
                    var $checkbox = $listItem.find('#' + _this7.settingValueIdPrefix + setting);

                    // Activate checkbox.
                    if (!$checkbox.is(':checked')) {
                        $checkbox.parent().trigger('click');
                    }

                    // Move to cache container.
                    $listItem.appendTo($sortedItems);
                };

                // Move active list items to the top bearing the sorting order in mind.
                columnSettings.forEach(settingIterator);

                // Prepend cached elements to item list.
                $sortedItems.children().prependTo(this.$settings);

                return this;
            }

            /**
             * Resets the column order and row height settings to the default.
             *
             * @return {OverviewSettingsModalController} Same instance for method chaining.
             *
             * @private
             */

        }, {
            key: '_setDefaultSettings',
            value: function _setDefaultSettings() {
                var _this8 = this;

                // Default values.
                var columnSettings = this.DEFAULT_COLUMN_SETTINGS;
                var rowHeight = this.DEFAULT_ROW_HEIGHT_SETTING;

                // Set column settings.
                // Cache container to temporarily hold all active list items in sorted order.
                // The children of this element will be prepended to the setting list item container to retain the
                // sorting order.
                var $sortedItems = $('<div/>');

                // Iterator function to prepend active list items to the top and activate the checkbox.
                var settingIterator = function settingIterator(setting) {
                    // List item ID.
                    var id = _this8.settingListItemIdPrefix + setting;

                    // Affected setting list item.
                    var $listItem = _this8.$settings.find('#' + id);

                    // Checkbox of affected list item.
                    var $checkbox = $listItem.find('#' + _this8.settingValueIdPrefix + setting);

                    // Activate checkbox.
                    if (!$checkbox.is(':checked')) {
                        $checkbox.parent().trigger('click');
                    }

                    // Move to cache container.
                    $listItem.appendTo($sortedItems);
                };

                // Deactivate all checkboxes.
                this.$settings.find(':checkbox').each(function (index, element) {
                    var $checkbox = $(element);

                    if ($checkbox.is(':checked')) {
                        $checkbox.parent().trigger('click');
                    }
                });

                // Move active list items to the top bearing the sorting order in mind.
                columnSettings.forEach(settingIterator);

                // Prepend cached elements to item list.
                $sortedItems.children().prependTo(this.$settings);

                // Set row height.
                this.$element.find(this.rowHeightValueSelector).val(rowHeight);

                return this;
            }
        }]);

        return OverviewSettingsModalController;
    }();

    exports.class = OverviewSettingsModalController;
})(jse.libs.overview_settings_modal_controller);
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
